<?php
/*
Plugin Name: Jídelní lístek restaurace
Description: Jednoduchý plugin pro správu týdenního jídelního lístku na jedné stránce.
Version: 1.0
Author: Petr Novák - digitalstrategy.cz
*/

// Zabezpečení proti přímému přístupu
if (!defined('ABSPATH')) {
    exit;
}

// Přidání položky do menu v administraci
add_action('admin_menu', 'jlr_add_admin_menu');
function jlr_add_admin_menu() {
    add_menu_page(
        'Jídelní lístek',
        'Jídelní lístek',
        'manage_options',
        'jidelni-listek',
        'jlr_render_admin_page',
        'dashicons-clipboard',
        6
    );
}

// Vykreslení stránky v administraci
function jlr_render_admin_page() {
    // Ukládání dat
    if (isset($_POST['jlr_nonce']) && wp_verify_nonce($_POST['jlr_nonce'], 'jlr_save_data')) {
        if (isset($_POST['jidelnicek'])) {
            $jidelnicek_data = array();
            foreach ($_POST['jidelnicek'] as $day => $items) {
                if (empty($items)) {
                    continue;
                }
                $jidelnicek_data[$day] = array_map(function($item) {
                    return array(
                        'type' => sanitize_text_field($item['type']),
                        'name' => sanitize_text_field($item['name']),
                        'alergeny' => sanitize_text_field($item['alergeny']),
                        'cena' => sanitize_text_field($item['cena']),
                    );
                }, $items);
            }
            update_option('jlr_jidelnicek', $jidelnicek_data);
            echo '<div class="notice notice-success is-dismissible"><p>Jídelní lístek byl uložen.</p></div>';
        }
    }

    $jidelnicek = get_option('jlr_jidelnicek', []);
    $days = ['Pondělí', 'Úterý', 'Středa', 'Čtvrtek', 'Pátek'];
    ?>
    <div class="wrap">
        <h1>Jídelní lístek</h1>
        <form method="post">
            <?php wp_nonce_field('jlr_save_data', 'jlr_nonce'); ?>
            <div id="jlr-jidelnicek-container">
                <?php foreach ($days as $day): ?>
                    <div class="jlr-day-section" id="day-<?php echo esc_attr($day); ?>">
                        <h2><?php echo esc_html($day); ?></h2>
                        <div class="jlr-items-container">
                            <?php if (!empty($jidelnicek[$day])): ?>
                                <?php foreach ($jidelnicek[$day] as $index => $item): ?>
                                    <div class="jlr-item">
                                        <input type="hidden" name="jidelnicek[<?php echo esc_attr($day); ?>][<?php echo esc_attr($index); ?>][type]" value="<?php echo esc_attr($item['type']); ?>">
                                        <input type="hidden" name="jidelnicek[<?php echo esc_attr($day); ?>][<?php echo esc_attr($index); ?>][name]" value="<?php echo esc_attr($item['name']); ?>">
                                        <input type="hidden" name="jidelnicek[<?php echo esc_attr($day); ?>][<?php echo esc_attr($index); ?>][alergeny]" value="<?php echo esc_attr($item['alergeny']); ?>">
                                        <input type="hidden" name="jidelnicek[<?php echo esc_attr($day); ?>][<?php echo esc_attr($index); ?>][cena]" value="<?php echo esc_attr($item['cena']); ?>">
                                        <strong><?php echo esc_html($item['type']); ?>:</strong> <?php echo esc_html($item['name']); ?> (<?php echo esc_html($item['cena']); ?> Kč, A:<?php echo esc_html($item['alergeny']); ?>)
                                        <button type="button" class="button button-small jlr-remove-item">Odebrat</button>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        <button type="button" class="button jlr-add-item-button" data-day="<?php echo esc_attr($day); ?>">Přidat jídlo</button>
                    </div>
                <?php endforeach; ?>
            </div>
            <p class="submit">
                <input type="submit" name="submit" id="submit" class="button button-primary" value="Uložit změny">
            </p>
        </form>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Přidání jídla
            document.querySelectorAll('.jlr-add-item-button').forEach(button => {
                button.addEventListener('click', function() {
                    const day = this.dataset.day;
                    const container = this.previousElementSibling;
                    const index = container.children.length;

                    const newItemHtml = `
                        <div class="jlr-item-form">
                            <h4>Nové jídlo pro ${day}</h4>
                            <p>Typ:
                                <select name="jidelnicek[${day}][${index}][type]">
                                    <option value="Polévka">Polévka</option>
                                    <option value="Hlavní jídlo">Hlavní jídlo</option>
                                </select>
                            </p>
                            <p>Název: <input type="text" name="jidelnicek[${day}][${index}][name]" placeholder="Název jídla"></p>
                            <p>Alergeny: <input type="text" name="jidelnicek[${day}][${index}][alergeny]" placeholder="Např. 1, 3, 7"></p>
                            <p>Cena: <input type="text" name="jidelnicek[${day}][${index}][cena]" placeholder="Cena v Kč"></p>
                            <button type="button" class="button button-small jlr-cancel-item">Zrušit</button>
                        </div>`;
                    
                    container.insertAdjacentHTML('beforeend', newItemHtml);
                });
            });

            // Zrušení přidávání
            document.addEventListener('click', function(e) {
                if (e.target.classList.contains('jlr-cancel-item')) {
                    e.target.closest('.jlr-item-form').remove();
                }
            });

            // Odebrání stávajícího jídla
            document.addEventListener('click', function(e) {
                if (e.target.classList.contains('jlr-remove-item')) {
                    if (confirm('Opravdu chcete odebrat toto jídlo?')) {
                        e.target.closest('.jlr-item').remove();
                    }
                }
            });
        });
    </script>
    <style>
        .jlr-day-section {
            border: 1px solid #ccc;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        .jlr-items-container {
            margin-bottom: 10px;
        }
        .jlr-item, .jlr-item-form {
            background: #f9f9f9;
            padding: 8px;
            border: 1px solid #eee;
            margin-bottom: 5px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .jlr-item-form {
            flex-direction: column;
            align-items: flex-start;
        }
        .jlr-item-form p {
            margin: 5px 0;
            width: 100%;
        }
    </style>
    <?php
}

// Funkce pro zobrazení jídelníčku pomocí shortcode
add_shortcode('jidelni-listek-restaurace', 'jlr_render_shortcode');
function jlr_render_shortcode() {
    $jidelnicek = get_option('jlr_jidelnicek', []);
    $output = '<div class="jlr-jidelnicek-shortcode">';
    $days_cz = ['Pondělí', 'Úterý', 'Středa', 'Čtvrtek', 'Pátek'];
    $days_en = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday'];

    foreach ($days_cz as $index => $day) {
        $output .= '<div class="jlr-day">';
        
        // Použijeme anglický název dne pro vytvoření data
        $day_en = $days_en[$index];
        $current_date = new DateTime('next ' . $day_en);

        // Zobrazíme český název dne a celé datum (včetně roku)
        $output .= '<h3>' . esc_html($day) . ' ' . $current_date->format('d.m.Y') . '</h3>';
        
        if (isset($jidelnicek[$day])) {
            $polevky = array_filter($jidelnicek[$day], function($item) {
                return $item['type'] === 'Polévka';
            });
            $hlavniJidla = array_filter($jidelnicek[$day], function($item) {
                return $item['type'] === 'Hlavní jídlo';
            });

            if (!empty($polevky)) {
                $output .= '<h4>Polévka:</h4><ul class="jlr-item-list">';
                foreach ($polevky as $item) {
                    $name = esc_html($item['name']);
                    $cena = esc_html($item['cena']);
                    $alergeny = esc_html($item['alergeny']);
                    
                    $output .= '<li>';
                    $output .= '<span class="jlr-name">' . $name . '</span>';
                    $output .= '&nbsp;<span style="font-weight: bold; font-size: 16px; color: #e74c3c;">' . $cena . ' Kč</span>';
                    if (!empty($alergeny)) {
                        $output .= '<span style="font-style: italic;"><br>Alergeny: ' . $alergeny . '</span>';
                    }
                    $output .= '</li>';
                }
                $output .= '</ul>';
            }

            if (!empty($hlavniJidla)) {
                $output .= '<h4>Hlavní jídlo:</h4><ul class="jlr-item-list">';
                foreach ($hlavniJidla as $item) {
                    $name = esc_html($item['name']);
                    $cena = esc_html($item['cena']);
                    $alergeny = esc_html($item['alergeny']);

                    $output .= '<li>';
                    $output .= '<span class="jlr-name">' . $name . '</span>';
                    $output .= '&nbsp;<span style="font-weight: bold; font-size: 16px; color: #e74c3c;">' . $cena . ' Kč</span>';
                    if (!empty($alergeny)) {
                       $output .= '<span style="font-style: italic;"><br>Alergeny: ' . $alergeny . '</span>';
                    }
                    $output .= '</li>';
                }
                $output .= '</ul>';
            }
        } else {
            $output .= '<p>Jídelníček pro tento den není dostupný.</p>';
        }
        $output .= '</div>';
    }
    $output .= '</div>';
    return $output;
}

